#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "offermon/memcache.h"
#include "offermon/tracker_static.h"


extern struct event_base *base;
struct timeval _time_elapsed1;
void time_elapsed_start()
{
	struct timezone tz;
	gettimeofday(&_time_elapsed1,&tz);

}

void time_elapsed_end(const char *name)
{
	struct timezone tz;
	struct timeval _time_elapsed2;
	struct timeval _time_elapsed3;
	gettimeofday(&_time_elapsed2,&tz);
	_time_elapsed3.tv_sec=_time_elapsed2.tv_sec-_time_elapsed1.tv_sec;
	if (_time_elapsed2.tv_usec>=_time_elapsed1.tv_usec) {
		_time_elapsed3.tv_usec=_time_elapsed2.tv_usec-_time_elapsed1.tv_usec;
	} else {
		_time_elapsed3.tv_sec--;
		_time_elapsed3.tv_usec=1000000+_time_elapsed2.tv_usec-_time_elapsed1.tv_usec;
	}
	printf("time_elapsed(%s) = %d.%03d\n",name,_time_elapsed3.tv_sec,_time_elapsed3.tv_usec/1000);
}

void _load__regions_done				(int fd, short event_type, void *arg) { Memcache *m=(Memcache*)arg; if (!m->db_done()) return; m->load__regions_done(); }
void _load__cpa_network_engines_done	(int fd, short event_type, void *arg) { Memcache *m=(Memcache*)arg; if (!m->db_done()) return; m->load__cpa_network_engines_done(); }
void _load__offers_done					(int fd, short event_type, void *arg) { Memcache *m=(Memcache*)arg; if (!m->db_done()) return; m->load__offers_done(); }
void _load__user_postbacks_done			(int fd, short event_type, void *arg) { Memcache *m=(Memcache*)arg; if (!m->db_done()) return; m->load__user_postbacks_done(); }
void _load__default_pricings_done		(int fd, short event_type, void *arg) { Memcache *m=(Memcache*)arg; if (!m->db_done()) return; m->load__default_pricings_done(); }
void _load__incent_sources_done			(int fd, short event_type, void *arg) { Memcache *m=(Memcache*)arg; if (!m->db_done()) return; m->load__incent_sources_done(); }
void _load__smartlinks_done				(int fd, short event_type, void *arg) { Memcache *m=(Memcache*)arg; if (!m->db_done()) return; m->load__smartlinks_done(); }
void _load__banners_done				(int fd, short event_type, void *arg) { Memcache *m=(Memcache*)arg; if (!m->db_done()) return; m->load__banners_done(); }




/**
 * Method: Memcache::new
 */
Memcache::Memcache()
{
	ptr_src=NULL;

	engines=new IntHash();
	engines->add_ref();

	offers=new IntHash();
	offers->add_ref();

	regions=new IntHash();
	regions->add_ref();

	user_postbacks=new IntHash();
	user_postbacks->add_ref();

	smartlinks=new IntHash();
	smartlinks->add_ref();

	banners=new IntHash();
	banners->add_ref();

	incent_sources=new IntHash();
	incent_sources->add_ref();

	default_pricings=new Array();
	default_pricings->add_ref();

	min_smartlink_id=0;
}

/**
 * Method: Memcache::delete
 */
Memcache::~Memcache()
{
	cleanup();
	engines			->del_ref();
	offers			->del_ref();
	regions			->del_ref();
	user_postbacks	->del_ref();
	smartlinks		->del_ref();
	banners			->del_ref();
	incent_sources	->del_ref();
	default_pricings->del_ref();
}

/**
 * Method: Memcache::cleanup
 */
void Memcache::cleanup()
{
	struct timeval tv;
	struct timezone tz;
	struct tm *p;
	gettimeofday(&tv,&tz);
   	p = localtime(&tv.tv_sec);
	strftime(last_update,32, "%Y-%m-%d %H:%M:%S", p);
	
/*	for (map<int64,CpaNetworkEngine*>::iterator iter1=engines.begin();iter1!=engines.end();iter1++) {
		if (iter1->second) delete iter1->second;
	}
	engines.clear();
	for (map<int64,Offer*>::iterator iter2=offers.begin();iter2!=offers.end();iter2++) {
		if (iter2->second) delete iter2->second;
	}
	offers.clear();
	for (map<int64,UserPostbacks*>::iterator iter3=user_postbacks.begin();iter3!=user_postbacks.end();iter3++) {
		if (iter3->second) delete iter3->second;
	}
	user_postbacks.clear();*/
	engines			->empty();
	offers			->empty();
	regions			->empty();
	user_postbacks	->empty();
	smartlinks		->empty();
	banners			->empty();
	incent_sources	->empty();
	default_pricings->empty();
	min_smartlink_id=0;
	if (ptr_src) free(ptr_src);
	ptr_src=NULL;
}

/**
 * Method: Memcache::get_offer
 */
OfferInfo *Memcache::get_offer(int64 offer_id,int region_id,int device_id,int affiliate_id)
{
	Offer *o=dynamic_cast<Offer*>(offers->get(offer_id));
	if (!o) return NULL;
	if (o->rr_offers) {
//		printf("rr_offer(%ld)=%ld\n",offer_id,o->rr_offers[o->rr_offer_pos]);
		offer_id=o->rr_offers[o->rr_offer_pos];
		o->rr_offer_pos++;
		if (o->rr_offer_pos==o->rr_offer_max) o->rr_offer_pos=0;
		o=dynamic_cast<Offer*>(offers->get(offer_id));
		if (!o) return NULL;
	}
//	printf("get_offer(%d) - device(%d)=%d region(%d)=%d\n",offer_id,device_id,o->get_device(device_id),region_id,o->get_region(region_id));
	if (device_id!=-1 && !o->get_device(device_id)) return NULL;
	if (region_id!=-1 && !o->get_region(region_id) && !o->get_region(634)) return NULL;
	OfferInfo *oi=new OfferInfo(o,dynamic_cast<CpaNetworkEngine*>(engines->get(o->engine_id)));
	return oi;
}

/**
 * Method: Memcache::load
 */
bool Memcache::load(char *ptr,int len)
{
	cleanup();

	ptr_src=ptr;
	ReadStream *s=new ReadStream((unsigned char*)ptr,len);
	char *str;
	int k,v,i;
	max_region_id=0;
	Offer*				offer;
	CpaNetworkEngine*	engine;
	UserPostbacks*		user_postback;
	Smartlink*			smartlink;
	Banner*				banner;
	DefaultPricing*		default_pricing;
	IncentSource*		incent_source;

//	int size				=s->read_int32();
	int cnt_regions			=s->read_int32();
	int cnt_engines			=s->read_int32();
	int cnt_offers			=s->read_int32();
	int cnt_user_postbacks	=s->read_int32();
	int cnt_smartlinks		=s->read_int32();
	int cnt_banners			=s->read_int32();
	int cnt_incent_sources	=s->read_int32();
	int cnt_default_pricings=s->read_int32();

	for (i=0;i<cnt_regions;i++) {
		k=s->read_int32();
		v=s->read_int32();
		regions->put(k,new Int64(v));
		if (max_region_id<v) max_region_id=v;
	}

	for (i=0;i<cnt_engines;i++) {
		engine=new CpaNetworkEngine();
		engine->load(s);
		engines->put(engine->id,engine);
	}

	for (i=0;i<cnt_offers;i++) {
		offer=new Offer();
		offer->init(max_region_id);
		offer->load(s);
		offers->put(offer->id,offer);
	}
	for (i=0;i<cnt_user_postbacks;i++) {
		user_postback=new UserPostbacks();
		user_postback->load(s);
		user_postbacks->put(user_postback->user_id,user_postback);
	}
	for (i=0;i<cnt_smartlinks;i++) {
		smartlink=new Smartlink();
		smartlink->load(s);
		smartlinks->put(smartlink->id,smartlink);
		if (min_smartlink_id==0) min_smartlink_id=smartlink->id;
	}
	for (i=0;i<cnt_banners;i++) {
		banner=new Banner();
		banner->load(s);
		Array *a=dynamic_cast<Array*>(banners->get(banner->offer_id));
		if (!a) {
			a=new Array();
			banners->put(banner->offer_id,a);
		}
		a->push(banner);
	}
	for (i=0;i<cnt_incent_sources;i++) {
		incent_source=new IncentSource();
		incent_source->load(s);
		incent_sources->put(incent_source->id,incent_source);
	}
	for (i=0;i<cnt_default_pricings;i++) {
		default_pricing=new DefaultPricing();
		default_pricing->load(s);
		default_pricings->push(default_pricing);

	}
	load__done();
	return true;
}

/**
 * Method: Memcache::save
 */
bool Memcache::save(WriteStream *s)
{
	int i,j;
	int banners_size=0;
	Array *a;
	for (i=0;i<banners->size;i++) {
		a=dynamic_cast<Array*>(banners->values[i]);
		banners_size+=a->size;
	}
	s->write_int32(regions->size);
	s->write_int32(engines->size);
	s->write_int32(offers->size);
	s->write_int32(user_postbacks->size);
	s->write_int32(smartlinks->size);
	s->write_int32(banners_size);
	s->write_int32(incent_sources->size);
	s->write_int32(default_pricings->size);
	for (i=0;i<regions->size;i++) {
		s->write_int32(regions->keys[i]);
		s->write_int32(dynamic_cast<Int64*>(regions->values[i])->value);
//		iter2->second->save(s);
	}
	for (i=0;i<engines->size;i++) {
		dynamic_cast<CpaNetworkEngine*>(engines->values[i])->save(s);
	}
	for (i=0;i<offers->size;i++) {
		dynamic_cast<Offer*>(offers->values[i])->save(s);
	}
	for (i=0;i<user_postbacks->size;i++) {
		dynamic_cast<UserPostbacks*>(user_postbacks->values[i])->save(s);
	}
	for (i=0;i<smartlinks->size;i++) {
		dynamic_cast<Smartlink*>(smartlinks->values[i])->save(s);
	}
	for (i=0;i<banners->size;i++) {
		a=dynamic_cast<Array*>(banners->values[i]);
		for (j=0;j<a->size;j++) {
			dynamic_cast<Banner*>(a->array[j])->save(s);
		}
	}
	for (i=0;i<incent_sources->size;i++) {
		dynamic_cast<IncentSource*>(incent_sources->values[i])->save(s);
	}
	for (i=0;i<default_pricings->size;i++) {
		dynamic_cast<DefaultPricing*>(default_pricings->array[j])->save(s);
	}
}

/**
 * Method: Memcache::pqexec_offers_list
 */
PGresult *Memcache::pqexec_offers_list(int64 offer_id,PGconn*__dbh)
{
	PGresult *res;
	char *param_values[1];
	char tmp[100];
	const char *sqlq=
		"select\n"
		"	o.id,\n"				//0
		"	cpa.engine_id,\n"		//1
		"	case when cpae.tracking_url_match is null then o.tracking_url else regexp_replace(o.tracking_url,cpae.tracking_url_match,cpae.tracking_url_replace,'') end as tracking_url,\n"	//2
//		"	'http://hit.offermon.com/?_server='||regexp_replace(case when cpae.tracking_url_match is null then o.tracking_url else regexp_replace(o.tracking_url,cpae.tracking_url_match,cpae.tracking_url_replace,'') end,'\\?','&') as tracking_url,\n"	//2
		"	(select string_agg(region_id::varchar,',') from Toffer_regions o_r where o_r.offer_id=o.id) as regions,\n"	//3
		"	(select string_agg(device_id::varchar,',') from Toffer_devices o_d where o_d.offer_id=o.id) as devices,\n"	//4
		"	a.name as app_name,\n"						//5
		"	ag_l.name as app_genre_name,\n"				//6
		"	a.description as app_description,\n"		//7
		"	a.installs as app_installs,\n"				//8
		"	a.rating_count as app_rating_count,\n"		//9
		"	a.rating_percent as app_rating_percent,\n"	//10
		"	a.cover_image as app_cover_image,\n"		//11
		"	case when exists (select 1 from Toffer_devices od,Tdevices d where d.is_ios=1 and d.id=od.device_id and od.offer_id=o.id) then 1 else 0 end as is_ios,\n" // 12
		"	case when exists (select 1 from Toffer_devices od,Tdevices d where d.is_android=1 and d.id=od.device_id and od.offer_id=o.id) then 1 else 0 end as is_android,\n" // 13
		"	(select string_agg(rr_offer_id::varchar,',') from Toffer_rrs orr where orr.offer_id=o.id) as rr,\n" // 14
		"	o.payout*c1.erate/c2.erate as payout_usd,\n" // 15
		"	o.payout,\n" // 16
		"	o.currency_id\n" // 17
		"from\n"
		"	Toffers o\n"
		"	left join Tapps a on o.store_id=a.store_id and a.language_id=1 and a.date_parsed is not null\n"
		"	left join Tapp_genres ag on a.genre_id=ag.id\n"
		"	left join Tapp_genres_l10n ag_l on ag.id=ag_l.fid_id and ag_l.l10n_id=1,\n"
		"	Tcpa_networks cpa,\n"
		"	Tcpa_network_engines cpae,\n"
		"	Tcurrencies c1,\n"
		"	Tcurrencies c2\n"
		"where (($1=0 and o.calc_disabled=0) or o.id=$1) and o.cpa_network_id=cpa.id and cpa.engine_id=cpae.id and o.currency_id=c1.id and c2.id=3\n"
		"order by o.id\n";
	sprintf(tmp,"%ld",offer_id);
	param_values[0]=tmp;
	if (offer_id==0) {
		if (!PQsendQueryParams(__dbh,sqlq,1,NULL,param_values,NULL,NULL,0)) {
			fprintf(stderr, "SELECT failed (Toffers):\n%s\n", PQerrorMessage(__dbh));
			exit(1);
		}
		return NULL;
	}
	/***** OFFERS *****/
	res = PQexecParams(__dbh,sqlq,1,NULL,param_values,NULL,NULL,0);
	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		fprintf(stderr, "SELECT failed (Toffers):\n%s\n", PQerrorMessage(__dbh));
		exit(1);
	}
	return res;
}

/**
 * Method: Memcache::load
 */
bool Memcache::load(PGconn *dbh)
{
	printf("Memcache::load(PGconn *dbh)\n");
	_dbh=dbh;
	cleanup();
	load__regions_start();
}

/**
 * Method: Memcache::load__done
 */
void Memcache::load__done()
{
	_dbh=NULL;
	ts->memcache_from_db_done();
}

/**
 * Method: Memcache:db_done
 */
bool Memcache::db_done()
{
	PQconsumeInput(_dbh);
	if (PQisBusy(_dbh)) {
//		printf("... got data (part)\n");
		return false;
	}
//	printf("... got data (complete)\n");

	event_del(_load_ev);
	event_free(_load_ev);
	_load_ev=NULL;
	return true;
}

/**
 * Method: Memcache::load__regions_start
 */
void Memcache::load__regions_start()
{
	pthread_mutex_lock(&dbh_mutex1);
	time_elapsed_start();
	if (!PQsendQuery(_dbh,"select id,code from Tregions order by code")) {
		fprintf(stderr, "SELECT failed (Tregions):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	if (PQstatus(_dbh) != CONNECTION_OK) {
		fprintf(stderr, "SELECT failed (Tregions):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	_load_ev=event_new(base,PQsocket(_dbh),EV_READ | EV_PERSIST, _load__regions_done,this);
	event_add(_load_ev, 0);
}

/**
 * Method: Memcache::load__regions_done
 */
void Memcache::load__regions_done()
{
	int64 id,id2;
	int i;
	char *code;
 	PGresult *res=PQgetResult(_dbh);
	time_elapsed_end("loading regions from SQL ");
	time_elapsed_start();
	max_region_id=0;
	while (res) {
		for (i=0;i<PQntuples(res);i++) {
			id=atol(PQgetvalue(res,i,0));
			code=PQgetvalue(res,i,1);
			id2=code[0]+(code[1]<<8);
			regions->put(id2,new Int64(id));
			if (max_region_id<id) max_region_id=id;
		}
		PQclear(res);
 		res=PQgetResult(_dbh);
	}
	pthread_mutex_unlock(&dbh_mutex1);
	time_elapsed_end("putting regions to memory");

	load__cpa_network_engines_start();
}

/**
 * Method: Memcache::load__cpa_network_engines_start
 */
void Memcache::load__cpa_network_engines_start()
{
	time_elapsed_start();
	pthread_mutex_lock(&dbh_mutex1);
	if (!PQsendQuery(_dbh,"select id,affre1,affre2 from Tcpa_network_engines")) {
		fprintf(stderr, "SELECT failed (Tcpa_network_engines):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	if (PQstatus(_dbh) != CONNECTION_OK) {
		fprintf(stderr, "SELECT failed (Tcpa_network_engines):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	_load_ev=event_new(base,PQsocket(_dbh),EV_READ | EV_PERSIST, _load__cpa_network_engines_done,this);
	event_add(_load_ev, 0);
}

/**
 * Method: Memcache::load__cpa_network_engines_done
 */
void Memcache::load__cpa_network_engines_done()
{
	int i;
	CpaNetworkEngine*	engine;
 	PGresult *res=PQgetResult(_dbh);
	time_elapsed_end("loading cpa networks engines from SQL ");
	time_elapsed_start();
	while (res) {
		for (i=0;i<PQntuples(res);i++) {
			engine=new CpaNetworkEngine();
			engine->load(res,i);
			engines->put(engine->id,engine);
		}
		PQclear(res);
		res=PQgetResult(_dbh);
	}
	pthread_mutex_unlock(&dbh_mutex1);
	time_elapsed_end("loading cpa networks engines to memory");

	load__offers_start();
}

/**
 * Method: Memcache::load__offers_start
 */
void Memcache::load__offers_start()
{
	time_elapsed_start();
	pthread_mutex_lock(&dbh_mutex1);
	Memcache::pqexec_offers_list(0,_dbh);
	if (PQstatus(_dbh) != CONNECTION_OK) {
		fprintf(stderr, "SELECT failed (Toffers):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	_load_ev=event_new(base,PQsocket(_dbh),EV_READ | EV_PERSIST, _load__offers_done,this);
	event_add(_load_ev, 0);
}

/**
 * Method: Memcache::load__offers_done
 */
void Memcache::load__offers_done()
{
	int i;
	Offer*				offer;
	struct timeval _time_elapsed1;
	struct timeval _time_elapsed2;
	struct timezone _tz;
	float _offers_put;

 	PGresult *res=PQgetResult(_dbh);
	time_elapsed_end("loading offers from SQL ");
	time_elapsed_start();
	_offers_put=0;
	while (res) {
		for (i=0;i<PQntuples(res);i++) {
			offer=new Offer();
			offer->init(max_region_id);
			offer->load(res,i);
			gettimeofday(&_time_elapsed1,&_tz);
			offers->put(offer->id,offer);
			gettimeofday(&_time_elapsed2,&_tz);
			_offers_put+=(_time_elapsed2.tv_sec-_time_elapsed1.tv_sec)+(_time_elapsed2.tv_usec-_time_elapsed1.tv_usec)/1000000.0;
		}
		PQclear(res);
		res=PQgetResult(_dbh);
	}
	pthread_mutex_unlock(&dbh_mutex1);
	time_elapsed_end("loading offers to memory");
	printf("time_elapsed(loading offers to memory) only offers->put taken %f seconds\n",_offers_put);

	load__user_postbacks_start();
}


/**
 * Method: Memcache::load__user_postbacks_start
 */
void Memcache::load__user_postbacks_start()
{
	time_elapsed_start();
	pthread_mutex_lock(&dbh_mutex1);
	if (!PQsendQuery(_dbh,"select user_id,code,url from Tuser_postbacks where url is not null order by user_id,code")) {
		fprintf(stderr, "SELECT failed (Tuser_postbacks):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	if (PQstatus(_dbh) != CONNECTION_OK) {
		fprintf(stderr, "SELECT failed (Tuser_postbacks):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	_load_ev=event_new(base,PQsocket(_dbh),EV_READ | EV_PERSIST, _load__user_postbacks_done,this);
	event_add(_load_ev, 0);
}

/**
 * Method: Memcache::load__user_postbacks_done
 */
void Memcache::load__user_postbacks_done()
{
	int i;
	int64 id,prev_id;
	char *code;
	char *url;
	UserPostbacks*		upbs;
 	PGresult *res=PQgetResult(_dbh);
	time_elapsed_end("loading user postbacks from SQL ");
	time_elapsed_start();
	upbs=NULL;
	prev_id=-1;
	while (res) {
		for (i=0;i<PQntuples(res);i++) {
			id=atol(PQgetvalue(res,i,0));
			code=PQgetvalue(res,i,1);
			url=PQgetvalue(res,i,2);
			if (id!=prev_id) {
				prev_id=id;
				upbs=new UserPostbacks();
				upbs->user_id=id;
				user_postbacks->put(upbs->user_id,upbs);
			}
			upbs->add(code,url);
		}
		PQclear(res);	
		res=PQgetResult(_dbh);
	}
	pthread_mutex_unlock(&dbh_mutex1);
	time_elapsed_end("loading user postbacks to memory");
	load__default_pricings_start();
}

/**
 * Method: Memcache::load__default_pricings_start
 */
void Memcache::load__default_pricings_start()
{
	time_elapsed_start();
	pthread_mutex_lock(&dbh_mutex1);
	if (!PQsendQuery(_dbh,"select id,coalesce(cpa_network_id,0),min_price,max_price,percent,percent_cpa from Tdefault_pricings where cpa_network_id is null order by min_price")) {
		fprintf(stderr, "SELECT failed (Tdefault_pricings):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	if (PQstatus(_dbh) != CONNECTION_OK) {
		fprintf(stderr, "SELECT failed (Tdefault_pricings):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	_load_ev=event_new(base,PQsocket(_dbh),EV_READ | EV_PERSIST, _load__default_pricings_done,this);
	event_add(_load_ev, 0);
}

/**
 * Method: Memcache::load__default_pricings_done
 */
void Memcache::load__default_pricings_done()
{
	int i;
	DefaultPricing*		default_pricing;
 	PGresult *res=PQgetResult(_dbh);
	time_elapsed_end("loading default pricings from SQL ");
	time_elapsed_start();
	while (res) {
		for (i=0;i<PQntuples(res);i++) {
			default_pricing=new DefaultPricing();
			default_pricing->load(res,i);
			default_pricings->push(default_pricing);
		}
		PQclear(res);
		res=PQgetResult(_dbh);
	}
	pthread_mutex_unlock(&dbh_mutex1);
	time_elapsed_end("loading default pricings to memory");
	load__incent_sources_start();
}

/**
 * Method: Memcache::load__incent_sources_start
 */
void Memcache::load__incent_sources_start()
{
	time_elapsed_start();
	pthread_mutex_lock(&dbh_mutex1);
	if (!PQsendQuery(_dbh,"select id,name from Tincent_sources where enabled=1")) {
		fprintf(stderr, "SELECT failed (Tincent_sources):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	if (PQstatus(_dbh) != CONNECTION_OK) {
		fprintf(stderr, "SELECT failed (Tincent_sources):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	_load_ev=event_new(base,PQsocket(_dbh),EV_READ | EV_PERSIST, _load__incent_sources_done,this);
	event_add(_load_ev, 0);
}

/**
 * Method: Memcache::load__incent_sources_done
 */
void Memcache::load__incent_sources_done()
{
	int i;
	IncentSource*		incent_source;
 	PGresult *res=PQgetResult(_dbh);
	time_elapsed_end("loading incent sources from SQL ");
	time_elapsed_start();
	while (res) {
		for (i=0;i<PQntuples(res);i++) {
			incent_source=new IncentSource();
			incent_source->load(res,i);
			incent_sources->put(incent_source->id,incent_source);
		}
		PQclear(res);	
		res=PQgetResult(_dbh);
	}
	pthread_mutex_unlock(&dbh_mutex1);
	time_elapsed_end("loading incent sources to memory");
	load__smartlinks_start();
}

/**
 * Method: Memcache::load__smartlinks_start
 */
void Memcache::load__smartlinks_start()
{
	time_elapsed_start();
	pthread_mutex_lock(&dbh_mutex1);
	if (!PQsendQuery(_dbh,
			"select\n"
			"	id,\n"
			"	user_id,\n"
			"	string_agg(offer_id::varchar,',' order by rnd) as offers,\n"
			"	string_agg(slo_id::varchar,',' order by rnd) as ids\n"
			"from (\n"
			"	select\n"
			"		sl.*,\n"
			"		o.id as offer_id,\n"
			"		slo.id as slo_id,\n"
			"		random() as rnd\n"
			"	from\n"
			"		Tsmartlinks sl,\n"
			"		Tsmartlink_offers slo,\n"
			"		Toffers o\n"
			"		left join Tlink_tests lt on o.last_test_id=lt.id\n"
			"	where\n"
			"		sl.id=slo.smartlink_id and\n"
			"		slo.offer_id=o.id and\n"
			"		o.calc_disabled=0 and\n"
			"		o.tracking_url is not null and\n"
			"		slo.enabled=1 and\n"
			"		(sl.hide_failed_link_checks=0 or lt.result_id in (5,8,11,9))\n"
			"	) tmp\n"
			"group by id,user_id\n"
		)) {
		fprintf(stderr, "SELECT failed (Tsmartlink_offers):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	};
	if (PQstatus(_dbh) != CONNECTION_OK) {
		fprintf(stderr, "SELECT failed (Tsmartlink_offers):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	_load_ev=event_new(base,PQsocket(_dbh),EV_READ | EV_PERSIST, _load__smartlinks_done, this);
	event_add(_load_ev, 0);
}

/**
 * Method: Memcache::load__smartlinks_done
 */
void Memcache::load__smartlinks_done()
{
	int i;
	Smartlink*			smartlink;
 	PGresult *res=PQgetResult(_dbh);
	time_elapsed_end("loading smartlinks from SQL ");
	time_elapsed_start();
	while (res) {
		for (i=0;i<PQntuples(res);i++) {
			smartlink=new Smartlink();
			smartlink->load(res,i);
			if (min_smartlink_id==0) min_smartlink_id=smartlink->id;
			smartlinks->put(smartlink->id,smartlink);
		}
		PQclear(res);	
		res=PQgetResult(_dbh);
	}
	pthread_mutex_unlock(&dbh_mutex1);
	time_elapsed_end("loading smartlinks to memory");
	load__banners_start();
}

/**
 * Method: Memcache::load__banners_start
 */
void Memcache::load__banners_start()
{
	time_elapsed_start();
	pthread_mutex_lock(&dbh_mutex1);
	if (!PQsendQuery(_dbh,
			"select\n"
			"	b.id,\n"
			"	b.offer_id,\n"
			"	b.type_id,\n"
			"	'/f-offermon'||Ffile_path(f_l.file_folder_id)||f_l.id||'.'||f_l.ext as logo_url,\n"
			"	'/f-offermon'||Ffile_path(f_i.file_folder_id)||f_i.id||'.'||f_i.ext as image_url,\n"
			"	b.name,\n"
			"	b.description\n"
			"from\n"
			"	Tbanners b,\n"
			"	Toffers o,\n"
			"	Tfiles f_l,\n"
			"	Tfiles f_i\n"
			"where\n"
			"	b.offer_id=o.id and\n"
			"	o.calc_disabled=0 and\n"
			"	b.logo_id=f_l.id and\n"
			"	b.image_id=f_i.id\n"
			"order by random()\n"
		)) {
		fprintf(stderr, "SELECT failed (Tbanners):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	if (PQstatus(_dbh) != CONNECTION_OK) {
		fprintf(stderr, "SELECT failed (Tbanners):\n%s\n", PQerrorMessage(_dbh));
		exit(1);
	}
	_load_ev=event_new(base,PQsocket(_dbh),EV_READ | EV_PERSIST, _load__banners_done,this);
	event_add(_load_ev, 0);
}

/**
 * Method: Memcache::load__banners_done
 */
void Memcache::load__banners_done()
{
	int i;
	Banner*				banner;
 	PGresult *res=PQgetResult(_dbh);
	time_elapsed_end("loading banners from SQL ");
	time_elapsed_start();
	while (res) {
		for (i=0;i<PQntuples(res);i++) {
			banner=new Banner();
			banner->load(res,i);
			Array *a=dynamic_cast<Array*>(banners->get(banner->offer_id));
			if (!a) {
				a=new Array();
				banners->put(banner->offer_id,a);
			}
			a->push(banner);
		}
		PQclear(res);
		res=PQgetResult(_dbh);
	}
	pthread_mutex_unlock(&dbh_mutex1);
	time_elapsed_end("loading banners to memory");
	load__done();
}

/**
 * Method: Memcache::calc_payout_aff
 */
float Memcache::calc_payout_aff(float payout_db,float payout_db_usd)
{
	DefaultPricing *dp;
	int i=0;
	for (i=0;i<default_pricings->size;i++) {
		dp=dynamic_cast<DefaultPricing*>(default_pricings->array[i]);
		if (payout_db_usd>=dp->min_price && payout_db_usd<=dp->max_price) {
			return payout_db*(100.0-dp->percent)/100.0;
		}
	}
	return payout_db*0.7;
}

/**
 * Method: Memcache::get_region
 */
int64 Memcache::get_region(int64 i)
{
//	printf("regions->size=%d\n",regions->size);
	Int64 *id=dynamic_cast<Int64*>(regions->get(i));
	if (!id) return 0;
	return id->value;
}

const char *Memcache::classname(void) { return "Memcache"; };
